<?php
// chat.php (text-only; OAuth Calendar event + Meet link; email kept; safe scopes; setter-style models)

// --------------------------------------------------------
// Requires
// --------------------------------------------------------
require_once __DIR__ . '/Helpers.php';
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/../vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// --------------------------------------------------------
// Headers, CORS, Preflight, Polyfills, Defaults
// --------------------------------------------------------

// Always return JSON
header('Content-Type: application/json; charset=utf-8');

// CORS (tighten Access-Control-Allow-Origin to your domain in prod)
if (!headers_sent()) {
  header('Access-Control-Allow-Origin: *');
  header('Access-Control-Allow-Methods: POST, OPTIONS');
  header('Access-Control-Allow-Headers: Content-Type, Authorization');
}

// Handle OPTIONS preflight cleanly
if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'OPTIONS') {
  http_response_code(204);
  exit;
}

// PHP 7 polyfill for str_contains
if (!function_exists('str_contains')) {
  function str_contains($haystack, $needle) {
    return $needle === '' || strpos((string)$haystack, (string)$needle) !== false;
  }
}

// Defaults for OpenAI env if not set in config.php
if (!isset($OPENAI_API_BASE) || !$OPENAI_API_BASE) {
  $OPENAI_API_BASE = 'https://api.openai.com/v1';
}
if (!isset($OPENAI_MODEL) || !$OPENAI_MODEL) {
  $OPENAI_MODEL = 'gpt-4o-mini';
}
if (!isset($OPENAI_API_KEY) || !$OPENAI_API_KEY) {
  json_response(['error' => 'Server is not configured: missing OPENAI_API_KEY'], 500);
}

// --------------------------------------------------------
// Basic IP Rate Limit (60 req / 10 min per IP)
// --------------------------------------------------------
$ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
$rateKey = sys_get_temp_dir() . '/chat_rl_' . md5($ip);
$now = time(); $win = 600; $limit = 60; $hits = 0;
if (file_exists($rateKey)) {
  $d = json_decode(@file_get_contents($rateKey), true);
  if ($d && $now - ($d['t'] ?? 0) < $win) { $hits = (int)($d['c'] ?? 0); }
}
$hits++;
@file_put_contents($rateKey, json_encode(['t' => $now, 'c' => $hits]));
if ($hits > $limit) {
  json_response(['error' => 'Too many requests. Please slow down.'], 429);
}

// --------------------------------------------------------
// Parse Request (JSON only — file uploads removed)
// --------------------------------------------------------
$session_id = '';
$message    = '';
$locale     = 'en';
$meta       = [];

$raw = file_get_contents('php://input');
$input = json_decode($raw, true);
if (!is_array($input)) {
  json_response(['error' => 'Invalid JSON body'], 400);
}

$session_id = sanitize($input['session_id'] ?? '');
$message    = sanitize($input['message'] ?? '');
$locale     = sanitize($input['locale'] ?? 'en');
$meta       = $input['meta'] ?? [];

if (!$session_id) { $session_id = 'sess_' . bin2hex(random_bytes(6)); }

// Require a message (no attachments supported)
if (!$message) {
  json_response(['error' => 'Message is required'], 422);
}

// --------------------------------------------------------
// Main Handler
// --------------------------------------------------------
try {
  // Create or find conversation
  $conversation_id = db_find_or_create_conversation($session_id, 'web');

  // Store user message
  $userContent = $message;
  db_insert_message($conversation_id, 'user', $userContent);

  // Prepare LLM messages (last 12 turns)
  $history = db_last_messages($conversation_id, 12);
  $messages = [];
  $messages[] = ['role' => 'system', 'content' => $SYSTEM_PROMPT]; // from config.php
  foreach ($history as $m) {
    $role = $m['role'] === 'assistant' ? 'assistant' : 'user';
    $messages[] = ['role' => $role, 'content' => (string)$m['content']];
  }

  // --- OpenAI Chat Completions call ---
  $payload = [
    'model'       => $OPENAI_MODEL,
    'messages'    => $messages,
    'temperature' => 0.6
  ];

  $url = rtrim($OPENAI_API_BASE, '/') . '/chat/completions';

  $ch = curl_init();
  curl_setopt_array($ch, [
    CURLOPT_URL            => $url,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST           => true,
    CURLOPT_HTTPHEADER     => [
      'Content-Type: application/json',
      'Authorization: Bearer ' . $OPENAI_API_KEY
    ],
    CURLOPT_POSTFIELDS     => json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE),
    CURLOPT_TIMEOUT        => 30
  ]);

  $raw   = curl_exec($ch);
  $cerr  = curl_error($ch);
  $code  = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);

  if ($raw === false) {
    error_log("LLM cURL error: $cerr");
    throw new Exception('LLM request failed: ' . $cerr);
  }

  $res = json_decode($raw, true);

  if ($code >= 400) {
    error_log("LLM HTTP $code: $raw");
    throw new Exception('LLM error: HTTP ' . $code);
  }

  if (!isset($res['choices'][0]['message']['content'])) {
    error_log("LLM unexpected payload: $raw");
    throw new Exception('LLM error: unexpected response payload');
  }

  $reply = (string)$res['choices'][0]['message']['content'];

  // Persist assistant reply
  db_insert_message($conversation_id, 'assistant', $reply);

  // Suggestions (simple intent heuristic)
  $lower = strtolower($message);
  $suggestions = ['Pricing', 'Book a demo', 'How are you?'];
  if (str_contains($lower, 'price') || str_contains($lower, 'cost')) {
    $suggestions = ['See plans', 'Book a demo', 'Features'];
  } elseif (str_contains($lower, 'book') || str_contains($lower, 'demo')) {
    $suggestions = ['Pick a time', 'Pricing', 'Integrations'];
  }

  // ------------------------------------------------------
  // "Book a demo" flow (save lead, calendar, email)
  // ------------------------------------------------------
  $eventLink = null; // Google Calendar event URL
  $meetLink  = null; // Google Meet link (if created)

  if (str_contains($lower, 'book a demo') || str_contains($lower, 'book demo')) {
    error_log("Received lead data: " . print_r($meta, true));

    $lead_name    = sanitize($meta['name']    ?? '');
    $lead_email   = sanitize($meta['email']   ?? '');
    $lead_phone   = sanitize($meta['phone']   ?? '');
    $lead_service = sanitize($meta['service'] ?? '');
    $lead_date    = sanitize($meta['date']    ?? '');
    $lead_time    = sanitize($meta['time']    ?? '');

    $lead_id = null;
    if ($lead_email && $lead_service) {
      $lead_id = db_find_lead_by_email_and_service($lead_email, $lead_service);
    }

    if ($lead_id) {
      db_update_lead($lead_id, $lead_name, $lead_email, $lead_phone, $lead_service, $lead_date, $lead_time);
    } else {
      $lead_id = db_insert_lead($conversation_id, $lead_name, $lead_email, $lead_phone, $lead_service, $lead_date, $lead_time);
    }

    if ($lead_id) error_log("Lead inserted/updated successfully: " . $lead_id);
    else error_log("Failed to insert/update lead.");

    if ($lead_date && $lead_time) {
      try {
        [$eventLink, $meetLink] = sendToGoogleCalendar(
          $lead_name, $lead_email, $lead_phone, $lead_service, $lead_date, $lead_time
        );
      } catch (\Throwable $e) {
        error_log("sendToGoogleCalendar failed: " . $e->getMessage());
      }
    } else {
      error_log("Calendar not created: missing date/time");
    }

    try {
      // EMAIL IS KEPT — same as before, but now passes event + meet links
      sendConfirmationEmail(
        $lead_name, $lead_email, $lead_phone, $lead_service, $lead_date, $lead_time, $eventLink, $meetLink
      );
    } catch (\Throwable $e) {
      error_log("sendConfirmationEmail failed: " . $e->getMessage());
    }

    if ($eventLink) {
      $reply .= "\n\nDemo booked successfully."
              . ($meetLink ? " Join via Google Meet: {$meetLink}" : "")
              . " View it on Google Calendar: {$eventLink}";
    } else {
      $reply .= "\n\nDemo booked successfully. We'll send a confirmation email soon.";
    }
  }

  // Final JSON response
  json_response([
    'session_id'  => $session_id,
    'reply'       => $reply,
    'suggestions' => $suggestions,
    'event_link'  => $eventLink,
    'meet_link'   => $meetLink,
  ]);

} catch (Exception $e) {
  error_log("chat.php exception: " . $e->getMessage());
  json_response(['error' => $e->getMessage()], 500);
}

// ========================================================
// Email + Calendar helpers (best-effort; do not throw)
// ========================================================
function sendConfirmationEmail($name, $email, $phone, $service, $date, $time, $eventLink = null, $meetLink = null) {
  $mail = new PHPMailer(true);
  try {
    $mail->isSMTP();
    $mail->Host = 'rovixai.com';
    $mail->SMTPAuth = true;
    $mail->Username = 'info@rovixai.com';
    $mail->Password = 'Limitless@#2025AI'; // TODO: move to env var!
    $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
    $mail->Port = 587;

    $mail->setFrom('info@rovixai.com', 'Rovix AI Assistant');
    if ($email) $mail->addAddress($email);

    $mail->isHTML(true);
    $mail->Subject = 'Your Rovix AI Demo is Scheduled 🎉';

    $safeName    = htmlspecialchars($name ?? '');
    $safePhone   = htmlspecialchars($phone ?? '');
    $safeService = htmlspecialchars($service ?? '');
    $safeDate    = htmlspecialchars($date ?? '');
    $safeTime    = htmlspecialchars($time ?? '');

    $meetLine = $meetLink
      ? '<p><strong>Your Google Meet link:</strong> <a href="' . htmlspecialchars($meetLink) . '" target="_blank" rel="noreferrer">' . htmlspecialchars($meetLink) . '</a></p>'
      : '';

    $eventLine = $eventLink
      ? '<p><strong>Add to Calendar:</strong> <a href="' . htmlspecialchars($eventLink) . '" target="_blank" rel="noreferrer">View in Google Calendar</a></p>'
      : '';

    // Polished email body
    $mail->Body = '
      <div style="font-family:Inter,Segoe UI,Roboto,Arial,sans-serif;line-height:1.6;color:#111">
        <p>Hi ' . $safeName . ',</p>
        <p><strong>Thank you for booking a demo with Rovix AI!</strong> 🎉</p>
        ' . $meetLine . '
        <p>We\'re excited to show you how our advanced AI solutions can simplify and supercharge your real estate journey.</p>

        <h3 style="margin:16px 0 8px">Appointment Details:</h3>
        <p><strong>Name:</strong> ' . $safeName . '<br>
           <strong>Service:</strong> ' . $safeService . '<br>
           <strong>Phone:</strong> ' . $safePhone . '</p>

        <p>📅 Your demo has been scheduled for <strong>' . $safeDate . '</strong> at <strong>' . $safeTime . '</strong> (Asia/Kolkata).</p>

        ' . $eventLine . '

        <p>If you have any questions or need to reschedule, reply to this email or contact us at <a href="mailto:info@rovixai.com">info@rovixai.com</a>.</p>

        <p>Looking forward to speaking with you!<br>— The Rovix AI Team</p>
      </div>
    ';

    if ($email) { $mail->send(); error_log("Confirmation email sent successfully to: $email"); }
  } catch (Exception $e) {
    error_log("Failed to send confirmation email: " . $mail->ErrorInfo);
  }
}

/**
 * OAuth client loader — uses token.json to refresh automatically.
 */
function getGoogleOAuthClient(): Google_Client {
  global $GOOGLE_OAUTH_CLIENT, $GOOGLE_OAUTH_TOKEN;

  $client = new Google_Client();
  $client->setApplicationName('Rovix AI');
  $client->setAuthConfig($GOOGLE_OAUTH_CLIENT);
  // Force array-of-strings for scopes (prevents implode() warnings)
  $client->setScopes(['https://www.googleapis.com/auth/calendar']);
  $client->setAccessType('offline');

  if (!is_file($GOOGLE_OAUTH_TOKEN)) {
    throw new Exception('OAuth token not found. Visit /ai-chatbot-starter/api/oauth_start.php to authorize.');
  }

  $token = json_decode(file_get_contents($GOOGLE_OAUTH_TOKEN), true);
  if (!is_array($token)) {
    throw new Exception('Invalid token file.');
  }

  $client->setAccessToken($token);

  if ($client->isAccessTokenExpired()) {
    $refreshed = $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
    if (isset($refreshed['error'])) {
      throw new Exception('Failed to refresh token: ' . ($refreshed['error_description'] ?? $refreshed['error']));
    }
    // Merge & persist
    $newToken = array_merge($token, $refreshed);
    file_put_contents($GOOGLE_OAUTH_TOKEN, json_encode($newToken, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
    @chmod($GOOGLE_OAUTH_TOKEN, 0600);
    $client->setAccessToken($newToken);
  }

  return $client;
}

/**
 * Create a Google Calendar event (with Google Meet) using OAuth user.
 * Returns [event_link, meet_link]
 */
function sendToGoogleCalendar($name, $email, $phone, $service, $date, $time) {
  try {
    global $GOOGLE_CALENDAR_ID;

    // --- helpers ------------------------------------------------------------
    $dbg = function (string $label, $val) {
      if (is_array($val)) {
        error_log("$label: array(" . count($val) . ")");
      } elseif (is_object($val)) {
        error_log("$label: object(" . get_class($val) . ")");
      } else {
        error_log("$label: " . var_export($val, true));
      }
    };

    $normalizeArrays = function (Google_Service_Calendar_Event $ev) {
      // Force known collection fields to arrays (PHP 8-safe)
      if (!is_array($ev->getAttendees()))   { $ev->setAttendees([]); }
      if (!is_array($ev->getRecurrence()))  { $ev->setRecurrence([]); }
      if (!is_array($ev->getAttachments())) { $ev->setAttachments([]); }

      // Reminders: either omit or ensure overrides is array
      $rem = $ev->getReminders();
      if ($rem instanceof Google_Service_Calendar_EventReminders) {
        // If you set reminders, make sure overrides is an array
        if (!is_array($rem->getOverrides())) {
          $rem->setOverrides([]);           // keep empty array
          $rem->setUseDefault(false);       // be explicit
          $ev->setReminders($rem);
        }
      }
    };

    // --- Google client/service ---------------------------------------------
    $client    = getGoogleOAuthClient();
    $cal       = new Google_Service_Calendar($client);

    // --- Build start/end in Asia/Kolkata -----------------------------------
    $tz = new DateTimeZone('Asia/Kolkata');
    $startDT = DateTime::createFromFormat('Y-m-d H:i', "$date $time", $tz);
    if (!$startDT) {
      throw new Exception("Invalid date/time: '$date $time' (expected Y-m-d and H:i)");
    }
    $endDT = (clone $startDT)->modify('+1 hour');

    $startObj = new Google_Service_Calendar_EventDateTime();
    $startObj->setDateTime($startDT->format(DateTime::RFC3339));
    $startObj->setTimeZone('Asia/Kolkata');

    $endObj = new Google_Service_Calendar_EventDateTime();
    $endObj->setDateTime($endDT->format(DateTime::RFC3339));
    $endObj->setTimeZone('Asia/Kolkata');

    // --- Event --------------------------------------------------------------
    $event = new Google_Service_Calendar_Event();
    $event->setSummary('Demo with ' . $name);
    $event->setLocation('Online');
    $event->setDescription("Demo for $service service.\nPhone: $phone\nEmail: $email");
    $event->setStart($startObj);
    $event->setEnd($endObj);

    // Always initialize collection fields to arrays
    $event->setAttendees([]);    // must not be null on PHP8
    $event->setRecurrence([]);   // ditto
    $event->setAttachments([]);  // ditto

    if (!empty($email)) {
      $att = new Google_Service_Calendar_EventAttendee();
      $att->setEmail($email);
      // Replace the empty array with one attendee
      $event->setAttendees([$att]);
    }

    // OPTIONAL: only set reminders if you really need them (kept off to avoid null overrides)
    // $rem = new Google_Service_Calendar_EventReminders();
    // $rem->setUseDefault(false);
    // $rem->setOverrides([]); // keep array even if empty
    // $event->setReminders($rem);

    // --- Ask for Google Meet ------------------------------------------------
    $requestId = 'req-' . bin2hex(random_bytes(6));
    $confKey   = new Google_Service_Calendar_ConferenceSolutionKey();
    $confKey->setType('hangoutsMeet');

    $createReq = new Google_Service_Calendar_CreateConferenceRequest();
    $createReq->setRequestId($requestId);
    $createReq->setConferenceSolutionKey($confKey);

    $confData = new Google_Service_Calendar_ConferenceData();
    $confData->setCreateRequest($createReq);
    $event->setConferenceData($confData);

    // --- Final normalization & DEBUG before insert -------------------------
    $normalizeArrays($event);

    $dbg('DEBUG start->dateTime', $event->getStart() ? $event->getStart()->getDateTime() : null);
    $dbg('DEBUG end->dateTime',   $event->getEnd() ? $event->getEnd()->getDateTime() : null);
    $dbg('DEBUG attendees',       $event->getAttendees());
    $dbg('DEBUG recurrence',      $event->getRecurrence());
    $dbg('DEBUG attachments',     $event->getAttachments());
    $dbg('DEBUG reminders set?',  $event->getReminders() ? 'yes' : 'no');
    $dbg('DEBUG conferenceData?', $event->getConferenceData() ? 'yes' : 'no');

    // --- Insert -------------------------------------------------------------
    $calendarId = $GOOGLE_CALENDAR_ID ?: 'primary';
    $event = $cal->events->insert($calendarId, $event, [
      'conferenceDataVersion' => 1,
      // 'sendUpdates' => 'all',
    ]);

    // --- Extract links ------------------------------------------------------
    $eventLink = $event->getHtmlLink() ?: null;

    $meetLink = null;
    $cd = $event->getConferenceData();
    if ($cd) {
      $eps = $cd->getEntryPoints();
      if (is_array($eps)) {
        foreach ($eps as $ep) {
          if (strtolower($ep->getEntryPointType()) === 'video' && $ep->getUri()) {
            $meetLink = $ep->getUri();
            break;
          }
        }
      }
    }
    if (!$meetLink && method_exists($event, 'getHangoutLink')) {
      $meetLink = $event->getHangoutLink() ?: null;
    }

    error_log("Event created: link=" . ($eventLink ?? 'none') . " meet=" . ($meetLink ?? 'none'));
    return [$eventLink, $meetLink];

  } catch (Throwable $e) {
    // Full diagnostics
    error_log("Error creating calendar event (OAuth): " . $e->getMessage());
    if (method_exists($e, 'getTraceAsString')) {
      error_log($e->getTraceAsString());
    }
    return [null, null];
  }
}
